/*
 * Copyright (C) 2014 Yahoo Inc.
 */

/**
 * A placeholder id prefix which is attached to every node which doesn't have an id in order to track the cursor.
 */
var COMPOSE_CONTENT_ELEMENT_ID_PREFIX = "yMail_cursorElementTracker_";

/**
 * An object which controls the focusing of the cursor in the compose body.
 */
var composeContentFocusController = {
    composeContentNode: null,
    resetCaretInAccessibilityMode: true,

    /**
     * A method which starts the composeContentFocusController
     */
    init: function(composeContentNode) {
        this.composeContentNode = composeContentNode;
        document.addEventListener("selectionchange", this.handleDocumentSelectionChange.bind(this), false);
    },

    /**
     * A method which handles selection change on the document
     */
    handleDocumentSelectionChange: function(event) {
        if (yMailBridge.getCurrentlyFocusedElementId() === this.composeContentNode.id) {
            // Set the cursor offset from the last sibling node
            yMailBridge.setCurrentlyFocusedBodyNodeCursorOffset(this.getCaretPosition());

            // Talkback will change the caret position to the middle of the input field. Set the position
            // back to the beginning for the first time user focus on the body node.
            if (this.resetCaretInAccessibilityMode && window.accessibilityMode) {
                setBodyFocusAtPosition(event.target.id, "", 0);
                this.resetCaretInAccessibilityMode = false;
            }
        } else {
            this.resetCaretInAccessibilityMode = true;
        }
    },

    /**
     * Focus the Compose Body Editor Element, adding a selection
     * range if own is not already present.
     */
    focusContentNode: function() {
        var sel,
            range,
            stationeryEditEl,
            signatureEl;

        // If stationery is applied, focus should be inside the stationery compose area
        if (Stationery.containsStationery()) {
            stationeryEditEl = document.querySelector('.rte-stationery-core #middleBody');
            if (stationeryEditEl) {
                this.composeContentNode = stationeryEditEl;
            }
        } else {
            this.composeContentNode = document.querySelector('#' + COMPOSE_CONTENT_ELEMENT_ID);
        }

        this.composeContentNode.focus();

        // Focusing on the composeContentNode is not always enough
        // Need to make sure a proper selection range is present.
        // Otherwise, execCommand / getRangeAt will not succeed.
        sel = window.getSelection && window.getSelection();
        if (sel && sel.rangeCount === 0) {
            // Since, we do not have a current selection range,
            // put the cursor at the end of the current compose content.
            signatureEl = document.getElementById(SIGNATURE_ELEMENT_ID);
            range = document.createRange();
            if (signatureEl) {
                // set cursor before signature element
                range.setStartBefore(signatureEl);
            } else {
                // set cursor to end of compose content node
                range.setStart(this.composeContentNode, 0);
                range.setEnd(this.composeContentNode, this.composeContentNode.childNodes.length);
                range.collapse(false); // collapse to end
            }
            sel.addRange(range);
        }
    },

    /**
     * A function that calculates the cursor position based on the selection and range objects for a element.
     * @param: editableDiv: document element object for which the cursor position needs to be calculated.
     */
    getCaretPosition : function() {
        var caretPos = INVALID_CURSOR_OFFSET;
        var selection;
        var range;

        if (window.getSelection) {
            selection = window.getSelection();
            if (selection.rangeCount) {
                range = selection.getRangeAt(0);
                caretPos = range.startOffset;
            }
        } else if (document.selection && document.selection.createRange) {
            range = document.selection.createRange();

            if (range.parentElement() == this.composeContentNode) {
                var tempEl = document.createElement("span");
                this.composeContentNode.insertBefore(tempEl, this.composeContentNode.firstChild);

                var tempRange = range.duplicate();
                tempRange.moveToElementText(tempEl);
                tempRange.setEndPoint("EndToEnd", range);
                caretPos = tempRange.text.length;
            }
        }

        return caretPos;
    },

    /**
     * A function to get the id and text of the currently focused node.
     * If the node does not have an id, a random value is set as id.
     * @return elementId : id of the focused node.
     *         nodeText  : text of the node.
     */
    getFocusedNodeIdAndText : function() {
        var node = document.getSelection().anchorNode;
        var nodeData = {};
        nodeData.elementId = null;
        nodeData.nodeText = null;

        if (node) {
            var element = node.nodeType == Node.TEXT_NODE ? node.parentNode : node;
            // Check for duplicate id
            if (!element.hasAttribute("id") ||
                (document.querySelector("#" + element.getAttribute("id")) && document.querySelectorAll("#" + element.getAttribute("id")).length > 1)) {
                element.setAttribute("id", COMPOSE_CONTENT_ELEMENT_ID_PREFIX + Date.now());
                syncComposeContent(this.composeContentNode, false);
            }
            nodeData.elementId = element.id;
            nodeData.nodeText = node.textContent;
        }

        return nodeData;
    },

    /**
     * Sets the cursor at the position for elementId
     * @param elementId: the id of the element on which cursor should be set.
     * @param nodeText: text of the node.
     * @param position: position of the cursor in the node.
     */
    setBodyFocusAtPosition : function(elementId, nodeText, position) {
        var nodeToFocus = document.getElementById(elementId);
        var docRange = document.createRange();

        if (nodeToFocus) {
            nodeIndex = 0;
            //if node has more child nodes, compare the nodeText to find the nodeIndex
            if (nodeToFocus.childNodes.length > 0) {
                for (i = 0; i < nodeToFocus.childNodes.length; i++) {
                    if (nodeToFocus.childNodes[i].textContent == nodeText) {
                        nodeIndex = i;
                        break;
                    }
                }
            }
            try {
                docRange.setStart(nodeToFocus.childNodes[nodeIndex], position);
            } catch (e) {
                // Fallback when focus is lost after stationary reapply, user has never focused anything
                var nodeToFocus;
                if (Stationery.containsStationery()) {
                    nodeToFocus = document.getElementById(STATIONERY_BODY_ID);
                } else {
                    nodeToFocus = document.getElementById(COMPOSE_CONTENT_ELEMENT_ID);
                }
                if (nodeToFocus.childNodes.length > 0) {
                    docRange.setStart(nodeToFocus.childNodes[0], 0);
                }
            }
        } else {
            // If nodeToFocus is not found, fallback to the stationery/compose main container.
            nodeToFocus = document.getElementById(STATIONERY_BODY_ID) || document.getElementById(COMPOSE_CONTENT_ELEMENT_ID);
            docRange.setStart(nodeToFocus.childNodes[0], 0);
        }

        var sel = window.getSelection();
        sel.removeAllRanges();
        sel.addRange(docRange);
    }
};
